#include <algorithm>
#include <iostream>
#include <vector>
#include <limits.h>
#include <stack>

using namespace std;

class SegmentTree
{
public:
    SegmentTree(const vector<int>& elements);
    /// <summary>
    /// Update all elements in range [l,r) with new_value if v[i] < new_value
    /// </summary>
    void update(int l, int r, int new_value);
    /// <summary>
    /// Return max in range [l,r)
    /// </summary>
    int max_query(int l, int r);
private:
    /// <summary>
    /// Update parents and the given node with new_value if 
    /// it is greater than the current value in nodes
    /// </summary>
    void update_parents(int node, int new_value);
    
    void propage_from_parents(int node);
    vector<int> tree;
    vector<int> delayed;
    int element_cnt;
};

/// <summary>
/// Paints the array in range [l,r] with  new_max if v[i] < new_max
/// </summary>
void paint_if_greater(vector<int>& v, int l, int r, int new_max)
{
    for (int i = l; i <= r; i++)
    {
        if (v[i] < new_max)
            v[i] = new_max;
    }
}

int Slamke(int N, int* A, int M, int* T, int* X, int* Y) {
    vector<int> temp(N);
    int solution = 0;
    
    for (int i = 0; i < N; i++)
    {
        if (A[i+1] == 1)
            temp[i] = 1;
    }
    SegmentTree max_reachable(temp);

    for (int query_index = 0; query_index < M; query_index++)
    {
        if (T[query_index+1] == 1) // l,r query
        {
            int l = X[query_index + 1] - 1;
            int r = Y[query_index + 1] - 1;
            int max_val = max_reachable.max_query(l, r + 1);
            max_reachable.update(l, r + 1, max_val);
        }
        else // Point query
        {
            int pos = X[query_index + 1] - 1;
            int value_at_pos = max_reachable.max_query(pos, pos + 1);
            solution = max(solution, value_at_pos);
            if (value_at_pos > 0)
            {
                max_reachable.update(pos, pos + 1, value_at_pos + 1);
            }
        }
    }
    return solution;
}

SegmentTree::SegmentTree(const vector<int>& elements)
{
    element_cnt = elements.size();
    tree = vector<int>(2 * element_cnt);
    delayed = tree;
    copy(elements.begin(), elements.end(), tree.begin() + element_cnt);
    for (int i = element_cnt-1; i > 0; i--)
        tree[i] = max(tree[2 * i], tree[2 * i + 1]);
    
}

void SegmentTree::update(int l, int r, int new_value)
{
    // Find nodes for interval [l,r) and update them and their parents
    l += element_cnt;
    r += element_cnt;
    while (l < r)
    {
        if (l % 2 == 1)
        {
            update_parents(l, new_value);
            delayed[l] = max(delayed[l],new_value);
            l++;
        }
        if (r % 2 == 1)
        {
            r--;
            update_parents(r, new_value);
            delayed[r] = max(delayed[r], new_value);
        }
        l /= 2;
        r /= 2;
    }
}

int SegmentTree::max_query(int l, int r)
{
    l += element_cnt;
    r += element_cnt;
    int max_val = INT_MIN;
    while (l < r)
    {
        if (l % 2 == 1)
        {
            propage_from_parents(l);
            max_val = max(tree[l], max_val);
            l++;
        }
        if (r % 2 == 1)
        {
            r--;
            propage_from_parents(r);
            max_val = max(tree[r], max_val);
        }
        l /= 2;
        r /= 2;
    }
    return max_val;
}
   
void SegmentTree::update_parents(int node, int new_value)
{
    for (int i = node; i > 0; i/=2)
    {
        if (tree[i] < new_value)
        {
            tree[i] = new_value;
        }
    }
}

void SegmentTree::propage_from_parents(int node)
{
    stack<int> parents;
    int max_val = INT_MIN;
    for (int i = node; i > 0; i /= 2)
    {
        max_val = max(max_val, delayed[i]);
        parents.push(i);
    }
    while (!parents.empty())
    {
        int x = parents.top();
        tree[x] = max(tree[x], max_val);
        parents.pop();
    }
}